<?php
// ams_at.php - Complete Attendance Management System
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database Configuration - UPDATE THESE WITH YOUR ACTUAL CREDENTIALS
define('DB_HOST', 'localhost');
define('DB_NAME', 'onesqeif_799'); // Replace with your actual database name
define('DB_USER', 'onesqeif_799'); // Replace with your actual username
define('DB_PASS', '!QJ{xhY5Dkz1'); // Replace with your actual password

// API Configuration
define('USER_API_URL', 'https://ams.onesbd.com/get_user.php');
define('API_TIMEOUT', 30);

// Attendance Rules
define('PRESENT_DEADLINE', '09:00:00');
define('LATE_DEADLINE', '09:15:00');
define('HALF_DAY_DEADLINE', '10:50:00');

class AttendanceSystem {
    private $conn;
    private $isConnected = false;
    
    public function __construct() {
        $this->initializeDatabase();
    }
    
    private function initializeDatabase() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
            $this->isConnected = true;
            return true;
        } catch(PDOException $e) {
            $this->isConnected = false;
            return false;
        }
    }
    
    public function isDatabaseConnected() {
        return $this->isConnected;
    }
    
    private function log($message, $type = "info") {
        $icons = [
            "success" => "✅",
            "error" => "❌",
            "warning" => "⚠️",
            "info" => "ℹ️"
        ];
        
        $icon = $icons[$type] ?? "📝";
        return "<div class='log-container log-$type'>$icon $message</div>";
    }
    
    private function ensureTables() {
        if (!$this->isConnected) return false;
        
        $tables = [
            'users' => "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id VARCHAR(50) UNIQUE NOT NULL,
                name VARCHAR(255) NOT NULL,
                phone VARCHAR(20),
                type ENUM('staff', 'student') NOT NULL,
                sync_status ENUM('pending', 'synced', 'failed') DEFAULT 'pending',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_user_id (user_id),
                INDEX idx_type (type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4",
            
            'attendance' => "CREATE TABLE IF NOT EXISTS attendance (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id VARCHAR(50) NOT NULL,
                check_in DATETIME NOT NULL,
                check_out DATETIME,
                status ENUM('P', 'L', 'HD', 'A') NOT NULL,
                hours_worked DECIMAL(4,2),
                sync_status ENUM('pending', 'synced', 'failed') DEFAULT 'synced',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_user_id (user_id),
                INDEX idx_date (check_in),
                INDEX idx_status (status)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4"
        ];
        
        $results = [];
        foreach ($tables as $name => $sql) {
            try {
                $this->conn->exec($sql);
                $results[$name] = $this->log("Table '$name' is ready", "success");
            } catch (Exception $e) {
                $results[$name] = $this->log("Error creating table '$name': " . $e->getMessage(), "error");
            }
        }
        return $results;
    }
    
    public function getUsersFromAPI() {
        // Simulated API response - Replace with actual API call
        $simulated_users = [
            ['uid' => 1, 'name' => 'Mst. Parul Akter', 'phone' => '017XXXXXXXX', 'type' => 'staff'],
            ['uid' => 2, 'name' => 'Mst. Rima Akter', 'phone' => '018XXXXXXXX', 'type' => 'staff'],
            ['uid' => 3, 'name' => 'Ripa Akter', 'phone' => '019XXXXXXXX', 'type' => 'staff'],
            ['uid' => 101, 'name' => 'Gazi Muhab Bin Ferdous', 'phone' => '017XXXXXXXX', 'type' => 'student'],
            ['uid' => 102, 'name' => 'Jamila Akter', 'phone' => '018XXXXXXXX', 'type' => 'student'],
            ['uid' => 103, 'name' => 'Abu Bokkar Siddik', 'phone' => '019XXXXXXXX', 'type' => 'student'],
            ['uid' => 104, 'name' => 'Md. Tawfiq', 'phone' => '017XXXXXXXX', 'type' => 'student'],
            ['uid' => 105, 'name' => 'Nusrat Jahan Sowrna', 'phone' => '018XXXXXXXX', 'type' => 'student'],
        ];
        
        return [
            'success' => true,
            'users' => $simulated_users,
            'message' => $this->log("API simulation successful - Received " . count($simulated_users) . " users", "success")
        ];
    }
    
    public function syncUsersToDatabase($users) {
        if (!$this->isConnected) {
            return [
                'success' => false,
                'count' => 0,
                'message' => $this->log("Database not connected", "error")
            ];
        }
        
        if (!is_array($users) || empty($users)) {
            return [
                'success' => false,
                'count' => 0,
                'message' => $this->log("No users data received", "error")
            ];
        }
        
        $synced = 0;
        $errors = 0;
        $logs = [];
        
        foreach ($users as $user) {
            try {
                $unique_id = $user['type'] . '_' . $user['uid'];
                $stmt = $this->conn->prepare("
                    INSERT INTO users (user_id, name, phone, type, sync_status) 
                    VALUES (?, ?, ?, ?, 'pending') 
                    ON DUPLICATE KEY UPDATE 
                    name = VALUES(name), 
                    phone = VALUES(phone), 
                    sync_status = 'pending'
                ");
                $stmt->execute([$unique_id, $user['name'], $user['phone'], $user['type']]);
                $synced++;
                $logs[] = $this->log("{$user['type']}: {$user['name']} (ID: {$user['uid']})", "success");
            } catch (Exception $e) {
                $errors++;
                $logs[] = $this->log("Error syncing {$user['name']}: " . $e->getMessage(), "error");
            }
        }
        
        return [
            'success' => $errors === 0,
            'count' => $synced,
            'errors' => $errors,
            'logs' => $logs,
            'message' => $this->log("Synced $synced users to database ($errors errors)", $errors === 0 ? "success" : "warning")
        ];
    }
    
    public function processAttendance($check_in, $check_out = null) {
        $check_time = date('H:i:s', strtotime($check_in));
        
        if ($check_time <= PRESENT_DEADLINE) {
            $status = 'P';
        } elseif ($check_time <= LATE_DEADLINE) {
            $status = 'L';
        } elseif ($check_time <= HALF_DAY_DEADLINE) {
            $status = 'HD';
        } else {
            $status = 'A';
        }
        
        $hours = null;
        if ($check_out) {
            $diff = strtotime($check_out) - strtotime($check_in);
            $hours = round($diff / 3600, 2);
        }
        
        return ['status' => $status, 'hours_worked' => $hours];
    }
    
    public function syncAttendance() {
        if (!$this->isConnected) {
            return [
                'success' => false,
                'count' => 0,
                'message' => $this->log("Database not connected", "error")
            ];
        }
        
        $sample_data = [
            ['user_id' => 'staff_1', 'check_in' => date('Y-m-d 08:45:00'), 'check_out' => date('Y-m-d 12:00:00')],
            ['user_id' => 'staff_2', 'check_in' => date('Y-m-d 09:10:00'), 'check_out' => date('Y-m-d 12:00:00')],
            ['user_id' => 'student_101', 'check_in' => date('Y-m-d 08:30:00'), 'check_out' => date('Y-m-d 12:00:00')],
            ['user_id' => 'student_102', 'check_in' => date('Y-m-d 09:20:00'), 'check_out' => date('Y-m-d 12:00:00')],
            ['user_id' => 'student_103', 'check_in' => date('Y-m-d 11:00:00'), 'check_out' => date('Y-m-d 12:00:00')],
        ];
        
        $processed = 0;
        $errors = 0;
        $logs = [];
        
        foreach ($sample_data as $record) {
            try {
                $attendance = $this->processAttendance($record['check_in'], $record['check_out']);
                
                $stmt = $this->conn->prepare("
                    INSERT INTO attendance (user_id, check_in, check_out, status, hours_worked) 
                    VALUES (?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $record['user_id'],
                    $record['check_in'],
                    $record['check_out'],
                    $attendance['status'],
                    $attendance['hours_worked']
                ]);
                
                $processed++;
                $logs[] = $this->log("{$record['user_id']}: {$record['check_in']} → <span class='status-{$attendance['status']}'>{$attendance['status']}</span> ({$attendance['hours_worked']}h)", "success");
            } catch (Exception $e) {
                $errors++;
                $logs[] = $this->log("Error processing {$record['user_id']}: " . $e->getMessage(), "error");
            }
        }
        
        return [
            'success' => $errors === 0,
            'count' => $processed,
            'errors' => $errors,
            'logs' => $logs,
            'message' => $this->log("Processed $processed attendance records ($errors errors)", $errors === 0 ? "success" : "warning")
        ];
    }
    
    public function getSystemStats() {
        if (!$this->isConnected) {
            return ['error' => 'Database not connected'];
        }
        
        $stats = [];
        
        try {
            // User statistics
            $stmt = $this->conn->query("SELECT COUNT(*) as total, type FROM users GROUP BY type");
            $stats['users'] = $stmt->fetchAll();
            
            // Attendance statistics
            $stmt = $this->conn->query("SELECT COUNT(*) as total FROM attendance");
            $stats['total_attendance'] = $stmt->fetch()['total'];
            
            $stmt = $this->conn->query("SELECT COUNT(*) as today FROM attendance WHERE DATE(check_in) = CURDATE()");
            $stats['today_attendance'] = $stmt->fetch()['today'];
            
            // Status distribution
            $stmt = $this->conn->query("SELECT status, COUNT(*) as count FROM attendance WHERE DATE(check_in) = CURDATE() GROUP BY status");
            $stats['status_dist'] = $stmt->fetchAll();
            
        } catch (Exception $e) {
            $stats['error'] = $e->getMessage();
        }
        
        return $stats;
    }
    
    public function testAttendanceRules() {
        $test_cases = [
            ['08:30:00', 'P'],
            ['09:05:00', 'L'],
            ['09:30:00', 'HD'],
            ['11:00:00', 'A']
        ];
        
        $logs = [];
        $logs[] = $this->log("Testing Attendance Rules", "info");
        
        foreach ($test_cases as $test) {
            $result = $this->processAttendance("2024-01-01 {$test[0]}");
            $icon = $result['status'] === $test[1] ? '✅' : '❌';
            $logs[] = $this->log("$icon {$test[0]} → <span class='status-{$result['status']}'>{$result['status']}</span> (Expected: {$test[1]})", 
                  $result['status'] === $test[1] ? "success" : "error");
        }
        
        return $logs;
    }
    
    public function runFullSync() {
        $logs = [];
        
        $logs[] = $this->log("🚀 Starting Full System Sync", "info");
        
        // Check database connection
        if (!$this->isConnected) {
            $logs[] = $this->log("❌ Database connection failed. Please check your credentials.", "error");
            return $logs;
        }
        
        $logs[] = $this->log("✅ Database connected successfully", "success");
        
        // Step 1: Ensure tables
        $tableResults = $this->ensureTables();
        $logs = array_merge($logs, array_values($tableResults));
        
        // Step 2: Sync users
        $apiResult = $this->getUsersFromAPI();
        $logs[] = $apiResult['message'];
        
        if ($apiResult['success']) {
            $syncResult = $this->syncUsersToDatabase($apiResult['users']);
            $logs[] = $syncResult['message'];
            $logs = array_merge($logs, $syncResult['logs']);
            
            // Step 3: Sync attendance
            $attendanceResult = $this->syncAttendance();
            $logs[] = $attendanceResult['message'];
            $logs = array_merge($logs, $attendanceResult['logs']);
            
            $logs[] = $this->log("🎉 Full Sync Completed Successfully!", "success");
        }
        
        return $logs;
    }
    
    public function getDashboardData() {
        if (!$this->isConnected) {
            return [
                'connected' => false,
                'error' => 'Database not connected'
            ];
        }
        
        $stats = $this->getSystemStats();
        return [
            'connected' => true,
            'stats' => $stats
        ];
    }
}

// HTML and UI Rendering
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AMS - Attendance Management System</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #2c3e50, #3498db);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
        }
        
        .header p {
            font-size: 1.2em;
            opacity: 0.9;
        }
        
        .control-panel {
            background: #f8f9fa;
            padding: 20px;
            border-bottom: 1px solid #dee2e6;
        }
        
        .button-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            justify-content: center;
        }
        
        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
            text-align: center;
        }
        
        .btn-primary { background: #007bff; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: #212529; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .btn-primary:hover { background: #0056b3; }
        .btn-success:hover { background: #218838; }
        .btn-warning:hover { background: #e0a800; }
        .btn-danger:hover { background: #c82333; }
        .btn-info:hover { background: #138496; }
        
        .content {
            padding: 30px;
        }
        
        .log-container {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin: 10px 0;
            border-left: 5px solid #007bff;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }
        
        .log-success { border-left-color: #28a745; background: #d4edda; }
        .log-error { border-left-color: #dc3545; background: #f8d7da; }
        .log-warning { border-left-color: #ffc107; background: #fff3cd; }
        .log-info { border-left-color: #17a2b8; background: #d1ecf1; }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 25px;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .stat-number {
            font-size: 2.5em;
            font-weight: bold;
            margin: 10px 0;
        }
        
        .stat-label {
            font-size: 1.1em;
            opacity: 0.9;
        }
        
        .table-container {
            overflow-x: auto;
            margin: 20px 0;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #dee2e6;
        }
        
        th {
            background: #343a40;
            color: white;
            font-weight: 600;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .status-P { color: #28a745; font-weight: bold; }
        .status-L { color: #ffc107; font-weight: bold; }
        .status-HD { color: #fd7e14; font-weight: bold; }
        .status-A { color: #dc3545; font-weight: bold; }
        
        .progress-bar {
            width: 100%;
            height: 10px;
            background: #e9ecef;
            border-radius: 5px;
            overflow: hidden;
            margin: 10px 0;
        }
        
        .progress {
            height: 100%;
            background: linear-gradient(90deg, #28a745, #20c997);
            transition: width 0.3s ease;
        }
        
        .connection-status {
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
            text-align: center;
            font-weight: bold;
        }
        
        .connected { background: #d4edda; color: #155724; }
        .disconnected { background: #f8d7da; color: #721c24; }
        
        .footer {
            text-align: center;
            padding: 20px;
            background: #f8f9fa;
            color: #6c757d;
            border-top: 1px solid #dee2e6;
        }
        
        @media (max-width: 768px) {
            .button-group {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏢 Attendance Management System</h1>
            <p>Alipura Model School - ZKTeco Biometric Integration</p>
        </div>
        
        <div class="control-panel">
            <div class="button-group">
                <a href="?action=full_sync" class="btn btn-primary">🔄 Full System Sync</a>
                <a href="?action=sync_users" class="btn btn-success">👥 Sync Users Only</a>
                <a href="?action=sync_attendance" class="btn btn-warning">🕐 Sync Attendance</a>
                <a href="?action=system_status" class="btn btn-info">📊 System Status</a>
                <a href="?action=test_rules" class="btn btn-primary">🧪 Test Rules</a>
                <a href="?" class="btn btn-danger">🔄 Refresh</a>
            </div>
        </div>
        
        <div class="content">
            <?php
            // Main execution
            $system = new AttendanceSystem();
            $dbConnected = $system->isDatabaseConnected();
            
            // Display connection status
            if ($dbConnected) {
                echo "<div class='connection-status connected'>✅ Database Connected Successfully</div>";
            } else {
                echo "<div class='connection-status disconnected'>❌ Database Connection Failed - Please check your credentials</div>";
            }
            
            if (isset($_GET['action'])) {
                switch ($_GET['action']) {
                    case 'full_sync':
                        $logs = $system->runFullSync();
                        foreach ($logs as $log) {
                            echo $log;
                        }
                        break;
                        
                    case 'sync_users':
                        if ($dbConnected) {
                            $apiResult = $system->getUsersFromAPI();
                            echo $apiResult['message'];
                            
                            if ($apiResult['success']) {
                                $syncResult = $system->syncUsersToDatabase($apiResult['users']);
                                echo $syncResult['message'];
                                foreach ($syncResult['logs'] as $log) {
                                    echo $log;
                                }
                            }
                        }
                        break;
                        
                    case 'sync_attendance':
                        if ($dbConnected) {
                            $result = $system->syncAttendance();
                            echo $result['message'];
                            foreach ($result['logs'] as $log) {
                                echo $log;
                            }
                        }
                        break;
                        
                    case 'test_rules':
                        $logs = $system->testAttendanceRules();
                        foreach ($logs as $log) {
                            echo $log;
                        }
                        break;
                        
                    case 'system_status':
                    default:
                        // Show system status
                        break;
                }
            }
            
            // Always show dashboard
            if ($dbConnected) {
                $dashboardData = $system->getDashboardData();
                if ($dashboardData['connected']) {
                    $stats = $dashboardData['stats'];
                    
                    echo "<div class='stats-grid'>";
                    
                    // Total Users
                    $totalUsers = 0;
                    if (isset($stats['users'])) {
                        foreach ($stats['users'] as $userType) {
                            $totalUsers += $userType['total'];
                        }
                    }
                    echo "<div class='stat-card'>
                        <div class='stat-label'>👥 Total Users</div>
                        <div class='stat-number'>$totalUsers</div>
                    </div>";
                    
                    // Today's Attendance
                    $todayAttendance = $stats['today_attendance'] ?? 0;
                    echo "<div class='stat-card'>
                        <div class='stat-label'>🕐 Today's Attendance</div>
                        <div class='stat-number'>$todayAttendance</div>
                    </div>";
                    
                    // Total Records
                    $totalRecords = $stats['total_attendance'] ?? 0;
                    echo "<div class='stat-card'>
                        <div class='stat-label'>📊 Total Records</div>
                        <div class='stat-number'>$totalRecords</div>
                    </div>";
                    
                    // System Status
                    echo "<div class='stat-card'>
                        <div class='stat-label'>🟢 System Status</div>
                        <div class='stat-number'>" . ($dbConnected ? 'Active' : 'Offline') . "</div>
                    </div>";
                    
                    echo "</div>";
                }
            }
            ?>
        </div>
        
        <div class="footer">
            <p>Attendance Management System &copy; 2024 Alipura Model School. Powered by ZKTeco Integration.</p>
        </div>
    </div>
</body>
</html>