<?php
require_once 'config.php';

class AttendanceSystem {
    private $conn;
    
    public function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . Config::DB_HOST . ";dbname=" . Config::DB_NAME,
                Config::DB_USER,
                Config::DB_PASS
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $e) {
            die("Database connection failed: " . $e->getMessage());
        }
    }
    
    // Fetch users from API - FIXED VERSION
    public function getUsersFromAPI() {
        echo "<h3>📡 Fetching Users from API</h3>";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::USER_API_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            echo "❌ API Error: HTTP $httpCode<br>";
            return false;
        }
        
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            echo "❌ JSON decode error<br>";
            return false;
        }
        
        echo "✅ API Success - Received " . count($data['data']) . " users<br>";
        return $data['data']; // Return only the users array
    }
    
    // Sync users to database - FIXED VERSION
    public function syncUsersToDatabase($users) {
        echo "<h3>👥 Syncing Users to Database</h3>";
        
        if (!is_array($users) || empty($users)) {
            echo "❌ No users data received<br>";
            return 0;
        }
        
        $synced = 0;
        foreach ($users as $user) {
            try {
                $user_id = $user['uid'];
                $name = $user['name'];
                $phone = $user['phone'];
                $type = $user['type']; // staff or student
                
                // Create unique user ID combining type and uid
                $unique_user_id = $type . '_' . $user_id;
                
                // Insert or update user in your existing users table
                $stmt = $this->conn->prepare("
                    INSERT INTO users (user_id, name, phone, type, sync_status) 
                    VALUES (?, ?, ?, ?, 'pending') 
                    ON DUPLICATE KEY UPDATE 
                    name = VALUES(name), 
                    phone = VALUES(phone), 
                    type = VALUES(type),
                    sync_status = 'pending',
                    updated_at = CURRENT_TIMESTAMP
                ");
                
                $stmt->execute([$unique_user_id, $name, $phone, $type]);
                $synced++;
                
                echo "✅ $type: $name (ID: $user_id)<br>";
                
            } catch (Exception $e) {
                echo "❌ Error syncing user {$user['name']}: " . $e->getMessage() . "<br>";
            }
        }
        
        echo "✅ Synced $synced users to database<br>";
        return $synced;
    }
    
    // Simulate ZKTeco device sync
    public function syncUsersToZKTeco($users) {
        echo "<h3>📱 Simulating ZKTeco Device Sync</h3>";
        
        $synced = 0;
        foreach ($users as $user) {
            // Simulate adding user to ZKTeco device
            $unique_user_id = $user['type'] . '_' . $user['uid'];
            echo "✅ ZKTeco: Added {$user['name']} (UID: {$user['uid']})<br>";
            $synced++;
            
            // Update sync status in database
            $this->updateUserSyncStatus($unique_user_id, 'synced');
        }
        
        echo "✅ $synced users prepared for ZKTeco device<br>";
        return $synced;
    }
    
    private function updateUserSyncStatus($user_id, $status) {
        $stmt = $this->conn->prepare("
            UPDATE users SET sync_status = ? WHERE user_id = ?
        ");
        $stmt->execute([$status, $user_id]);
    }
    
    // Process attendance with your rules
    public function processAttendance($check_in, $check_out = null) {
        $check_in_time = date('H:i:s', strtotime($check_in));
        
        // Calculate status based on your rules
        if ($check_in_time <= '09:00:00') {
            $status = 'P'; // Present
        } elseif ($check_in_time <= '09:15:00') {
            $status = 'L'; // Late
        } elseif ($check_in_time <= '10:50:00') {
            $status = 'HD'; // Half Day
        } else {
            $status = 'A'; // Absent
        }
        
        // Calculate hours worked if check_out provided
        $hours_worked = null;
        if ($check_out) {
            $check_in_ts = strtotime($check_in);
            $check_out_ts = strtotime($check_out);
            $hours_worked = round(($check_out_ts - $check_in_ts) / 3600, 2);
        }
        
        return [
            'status' => $status,
            'hours_worked' => $hours_worked
        ];
    }
    
    // Simulate attendance sync from ZKTeco to cPanel
    public function syncAttendanceFromZKTeco() {
        echo "<h3>🕐 Simulating Attendance Sync (ZKTeco → cPanel)</h3>";
        
        // This would normally come from ZKTeco device
        $sample_attendance = [
            ['user_id' => 'staff_2', 'check_in' => '2024-01-15 08:45:00', 'check_out' => '2024-01-15 12:00:00'],
            ['user_id' => 'student_1', 'check_in' => '2024-01-15 09:10:00', 'check_out' => '2024-01-15 12:00:00'],
            ['user_id' => 'student_4', 'check_in' => '2024-01-15 10:30:00', 'check_out' => '2024-01-15 12:00:00'],
            ['user_id' => 'staff_3', 'check_in' => '2024-01-15 11:15:00', 'check_out' => '2024-01-15 12:00:00'],
        ];
        
        $processed = 0;
        foreach ($sample_attendance as $record) {
            $attendance_data = $this->processAttendance($record['check_in'], $record['check_out']);
            
            // Insert into attendance table
            $stmt = $this->conn->prepare("
                INSERT INTO attendance (user_id, check_in, check_out, status, hours_worked, sync_status) 
                VALUES (?, ?, ?, ?, ?, 'synced')
            ");
            
            $stmt->execute([
                $record['user_id'],
                $record['check_in'],
                $record['check_out'],
                $attendance_data['status'],
                $attendance_data['hours_worked']
            ]);
            
            echo "✅ {$record['user_id']}: {$record['check_in']} → {$attendance_data['status']} ({$attendance_data['hours_worked']} hrs)<br>";
            $processed++;
        }
        
        echo "✅ Processed $processed attendance records<br>";
        return $processed;
    }
    
    // Test the system
    public function testAttendanceRules() {
        echo "<h3>⏰ Testing Attendance Rules</h3>";
        
        $test_cases = [
            ['08:30:00', 'P'],
            ['09:05:00', 'L'],
            ['09:30:00', 'HD'],
            ['11:00:00', 'A']
        ];
        
        foreach ($test_cases as $test) {
            $result = $this->processAttendance("2024-01-01 {$test[0]}");
            $icon = $result['status'] === $test[1] ? '✅' : '❌';
            echo "$icon {$test[0]} → {$result['status']} (Expected: {$test[1]})<br>";
        }
    }
    
    // Main sync function
    public function runCompleteSync() {
        echo "<h1>🔄 Complete Attendance System Sync</h1>";
        echo "<div style='background: #f5f5f5; padding: 20px; border-radius: 10px;'>";
        
        // Step 1: Test database
        echo "<h3>Step 1: Database Connection</h3>";
        echo "✅ Database connected successfully<br><br>";
        
        // Step 2: Test rules
        echo "<h3>Step 2: Attendance Rules Test</h3>";
        $this->testAttendanceRules();
        echo "<br>";
        
        // Step 3: Sync users from API to cPanel
        echo "<h3>Step 3: User Sync (cPanel ← API)</h3>";
        $users = $this->getUsersFromAPI();
        
        if ($users) {
            // Step 4: Sync to database
            $db_count = $this->syncUsersToDatabase($users);
            
            // Step 5: Sync to ZKTeco device
            echo "<h3>Step 4: User Sync (cPanel → ZKTeco)</h3>";
            $zk_count = $this->syncUsersToZKTeco($users);
            
            // Step 6: Sync attendance from ZKTeco
            echo "<h3>Step 5: Attendance Sync (ZKTeco → cPanel)</h3>";
            $attendance_count = $this->syncAttendanceFromZKTeco();
            
            echo "<h3 style='color: green;'>🎉 Sync Completed Successfully!</h3>";
            echo "<ul>";
            echo "<li>📊 Users synced to database: $db_count</li>";
            echo "<li>📱 Users prepared for ZKTeco: $zk_count</li>";
            echo "<li>🕐 Attendance records processed: $attendance_count</li>";
            echo "</ul>";
        }
        
        echo "</div>";
        
        // Show current data
        $this->showCurrentData();
    }
    
    // Show current data in database
    private function showCurrentData() {
        echo "<h3>📊 Current Data in Database</h3>";
        
        // Count users
        $stmt = $this->conn->query("SELECT COUNT(*) as count FROM users");
        $user_count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        // Count attendance
        $stmt = $this->conn->query("SELECT COUNT(*) as count FROM attendance");
        $attendance_count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        echo "Total users in database: $user_count<br>";
        echo "Total attendance records: $attendance_count<br>";
        
        // Show recent users
        $stmt = $this->conn->query("SELECT user_id, name, type FROM users ORDER BY created_at DESC LIMIT 5");
        $recent_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "<h4>Recent Users:</h4>";
        foreach ($recent_users as $user) {
            echo "• {$user['name']} ({$user['type']}) - {$user['user_id']}<br>";
        }
    }
}

// Run the system
if (isset($_GET['action'])) {
    $system = new AttendanceSystem();
    
    switch ($_GET['action']) {
        case 'sync_users':
            $users = $system->getUsersFromAPI();
            if ($users) {
                $system->syncUsersToDatabase($users);
                $system->syncUsersToZKTeco($users);
            }
            break;
            
        case 'sync_attendance':
            $system->syncAttendanceFromZKTeco();
            break;
            
        case 'full_sync':
        default:
            $system->runCompleteSync();
            break;
    }
} else {
    // Show control panel
    echo '<!DOCTYPE html>
    <html>
    <head>
        <title>Attendance System Control Panel</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; background: #f0f0f0; }
            .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 10px; }
            .button { display: inline-block; padding: 15px 30px; margin: 10px; background: #007bff; color: white; 
                     text-decoration: none; border-radius: 5px; font-size: 16px; }
            .button:hover { background: #0056b3; }
            .log { background: #f8f9fa; padding: 15px; border-radius: 5px; margin-top: 20px; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>🏢 Attendance System Control Panel</h1>
            <p>Alipura Model School - ZKTeco Integration</p>
            
            <div>
                <a href="?action=full_sync" class="button">🔄 Full Sync</a>
                <a href="?action=sync_users" class="button">👥 Sync Users Only</a>
                <a href="?action=sync_attendance" class="button">🕐 Sync Attendance Only</a>
            </div>
            
            <div class="log">
                <h3>Quick Stats:</h3>
                <p>Run a sync operation to see statistics here...</p>
            </div>
        </div>
    </body>
    </html>';
}
?>