<?php
require_once 'config.php';

class AttendanceSystemFixed {
    private $conn;
    
    public function __construct() {
        try {
            $this->conn = new PDO(
                "mysql:host=" . Config::DB_HOST . ";dbname=" . Config::DB_NAME,
                Config::DB_USER,
                Config::DB_PASS
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $e) {
            die("Database connection failed: " . $e->getMessage());
        }
    }
    
    // Fixed: Properly fetch and return users array
    public function getUsersFromAPI() {
        echo "<h3>📡 Fetching Users from API</h3>";
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => Config::USER_API_URL,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            echo "❌ API Error: HTTP $httpCode<br>";
            return false;
        }
        
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE || !isset($data['data'])) {
            echo "❌ JSON decode error or invalid format<br>";
            return false;
        }
        
        $users = $data['data'];
        echo "✅ API Success - Received " . count($users) . " users<br>";
        
        // Debug: Show first few users
        echo "<h4>First 5 Users Sample:</h4>";
        for ($i = 0; $i < min(5, count($users)); $i++) {
            echo "• {$users[$i]['name']} (UID: {$users[$i]['uid']}, Type: {$users[$i]['type']})<br>";
        }
        
        return $users;
    }
    
    // Fixed: Proper user synchronization
    public function syncUsersToDatabase($users) {
        echo "<h3>👥 Syncing Users to Database</h3>";
        
        if (!is_array($users) || empty($users)) {
            echo "❌ No users data received<br>";
            return 0;
        }
        
        $synced = 0;
        $errors = 0;
        
        foreach ($users as $user) {
            try {
                // Validate required fields
                if (!isset($user['uid']) || !isset($user['name']) || !isset($user['type'])) {
                    echo "❌ Skipping invalid user data<br>";
                    $errors++;
                    continue;
                }
                
                $user_id = $user['uid'];
                $name = $user['name'];
                $phone = $user['phone'] ?? '';
                $type = $user['type'];
                
                // Create unique user ID combining type and uid
                $unique_user_id = $type . '_' . $user_id;
                
                // Check if user already exists in zk_users table (if it exists)
                $user_exists = false;
                try {
                    $stmt = $this->conn->prepare("SELECT COUNT(*) FROM zk_users WHERE user_id = ?");
                    $stmt->execute([$unique_user_id]);
                    $user_exists = $stmt->fetchColumn() > 0;
                } catch (Exception $e) {
                    // zk_users table might not exist, continue with users table
                }
                
                // Use zk_users table if it exists, otherwise use users table
                $table_name = $user_exists ? 'zk_users' : 'users';
                
                // Insert or update user
                $stmt = $this->conn->prepare("
                    INSERT INTO $table_name (user_id, name, phone, type, sync_status) 
                    VALUES (?, ?, ?, ?, 'pending') 
                    ON DUPLICATE KEY UPDATE 
                    name = VALUES(name), 
                    phone = VALUES(phone), 
                    type = VALUES(type),
                    sync_status = 'pending',
                    updated_at = CURRENT_TIMESTAMP
                ");
                
                $stmt->execute([$unique_user_id, $name, $phone, $type]);
                
                if ($stmt->rowCount() > 0) {
                    echo "✅ $type: $name (ID: $user_id)<br>";
                    $synced++;
                } else {
                    echo "⚠️ No changes: $name<br>";
                }
                
            } catch (Exception $e) {
                echo "❌ Error syncing user '{$user['name']}': " . $e->getMessage() . "<br>";
                $errors++;
            }
        }
        
        echo "✅ Successfully synced $synced users to database<br>";
        if ($errors > 0) {
            echo "❌ $errors users had errors<br>";
        }
        
        return $synced;
    }
    
    // Test database tables
    public function testDatabaseTables() {
        echo "<h3>🗃️ Checking Database Tables</h3>";
        
        $tables = ['users', 'zk_users', 'attendance', 'student_attendance'];
        
        foreach ($tables as $table) {
            try {
                $stmt = $this->conn->query("SELECT COUNT(*) as count FROM $table");
                $count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
                echo "✅ $table: $count records<br>";
            } catch (Exception $e) {
                echo "❌ $table: Table doesn't exist or error: " . $e->getMessage() . "<br>";
            }
        }
    }
    
    // Calculate attendance status
    public function calculateAttendanceStatus($checkInTime) {
        $checkIn = date('H:i:s', strtotime($checkInTime));
        
        if ($checkIn <= '09:00:00') return 'P'; // Present
        if ($checkIn <= '09:15:00') return 'L'; // Late
        if ($checkIn <= '10:50:00') return 'HD'; // Half Day
        return 'A'; // Absent
    }
    
    // Test attendance rules
    public function testAttendanceRules() {
        echo "<h3>⏰ Testing Attendance Rules</h3>";
        
        $testCases = [
            '08:30:00' => 'P',
            '09:05:00' => 'L',
            '09:30:00' => 'HD',
            '11:00:00' => 'A'
        ];
        
        foreach ($testCases as $time => $expected) {
            $status = $this->calculateAttendanceStatus("2024-01-01 $time");
            $result = $status === $expected ? '✅' : '❌';
            echo "$result $time → $status (Expected: $expected)<br>";
        }
    }
    
    // Main sync function
    public function runCompleteSync() {
        echo "<h1>🔄 Complete Attendance System Sync</h1>";
        echo "<div style='background: #f5f5f5; padding: 20px; border-radius: 10px; margin: 20px 0;'>";
        
        // Step 1: Test database connection
        echo "<h3>Step 1: Database Connection</h3>";
        echo "✅ Database connected successfully<br><br>";
        
        // Step 2: Check tables
        $this->testDatabaseTables();
        echo "<br>";
        
        // Step 3: Test attendance rules
        echo "<h3>Step 2: Attendance Rules Test</h3>";
        $this->testAttendanceRules();
        echo "<br>";
        
        // Step 4: Fetch users from API
        echo "<h3>Step 3: Fetch Users from API</h3>";
        $users = $this->getUsersFromAPI();
        
        if ($users && count($users) > 0) {
            // Step 5: Sync to database
            echo "<h3>Step 4: Sync Users to Database</h3>";
            $synced_count = $this->syncUsersToDatabase($users);
            
            echo "<h3 style='color: green;'>🎉 Sync Completed Successfully!</h3>";
            echo "<ul>";
            echo "<li>📊 Total users from API: " . count($users) . "</li>";
            echo "<li>✅ Users synced to database: $synced_count</li>";
            echo "<li>👥 Staff members: " . count(array_filter($users, fn($u) => $u['type'] === 'staff')) . "</li>";
            echo "<li>🎓 Student members: " . count(array_filter($users, fn($u) => $u['type'] === 'student')) . "</li>";
            echo "</ul>";
            
            // Show summary
            $this->showSyncSummary($users);
            
        } else {
            echo "<h3 style='color: red;'>❌ Sync Failed - No users received from API</h3>";
        }
        
        echo "</div>";
    }
    
    // Show sync summary
    private function showSyncSummary($users) {
        echo "<h3>📊 Sync Summary</h3>";
        
        $staff_count = 0;
        $student_count = 0;
        
        foreach ($users as $user) {
            if ($user['type'] === 'staff') $staff_count++;
            if ($user['type'] === 'student') $student_count++;
        }
        
        echo "• Staff Members: $staff_count<br>";
        echo "• Student Members: $student_count<br>";
        echo "• Total Users: " . count($users) . "<br>";
        
        // Show recent synced users from database
        echo "<h4>Recently Synced Users:</h4>";
        try {
            $stmt = $this->conn->query("
                SELECT user_id, name, type, sync_status 
                FROM users 
                ORDER BY created_at DESC 
                LIMIT 10
            ");
            $recent_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($recent_users) > 0) {
                foreach ($recent_users as $user) {
                    $status_icon = $user['sync_status'] === 'synced' ? '✅' : '⏳';
                    echo "$status_icon {$user['name']} ({$user['type']}) - {$user['user_id']}<br>";
                }
            } else {
                echo "No users found in database<br>";
            }
        } catch (Exception $e) {
            echo "Error reading users: " . $e->getMessage() . "<br>";
        }
    }
}

// Handle the request
if (isset($_GET['action']) || isset($_POST['action'])) {
    $action = $_GET['action'] ?? $_POST['action'] ?? 'full_sync';
    
    $system = new AttendanceSystemFixed();
    
    switch ($action) {
        case 'sync_users':
            $users = $system->getUsersFromAPI();
            if ($users) {
                $system->syncUsersToDatabase($users);
            }
            break;
            
        case 'test_api':
            $users = $system->getUsersFromAPI();
            if ($users) {
                echo "<h3>API Response Details:</h3>";
                echo "<pre>" . print_r(array_slice($users, 0, 10), true) . "</pre>";
            }
            break;
            
        case 'full_sync':
        default:
            $system->runCompleteSync();
            break;
    }
} else {
    // Show control panel
    echo '<!DOCTYPE html>
    <html>
    <head>
        <title>Attendance System - Fixed Version</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 20px; background: #f0f0f0; }
            .container { max-width: 900px; margin: 0 auto; background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .button { display: inline-block; padding: 12px 25px; margin: 8px; background: #28a745; color: white; 
                     text-decoration: none; border-radius: 5px; font-size: 16px; font-weight: bold; }
            .button:hover { background: #218838; transform: translateY(-2px); }
            .button.test { background: #17a2b8; }
            .button.sync { background: #ffc107; color: black; }
            .log { background: #f8f9fa; padding: 15px; border-radius: 5px; margin-top: 20px; border-left: 4px solid #007bff; }
            .success { color: #28a745; }
            .error { color: #dc3545; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>🏢 Attendance System - Fixed Version</h1>
            <p class="success">Alipura Model School - ZKTeco Integration</p>
            
            <div>
                <a href="?action=full_sync" class="button">🔄 Full Sync</a>
                <a href="?action=sync_users" class="button sync">👥 Sync Users Only</a>
                <a href="?action=test_api" class="button test">🔍 Test API</a>
            </div>
            
            <div class="log">
                <h3>📋 Instructions:</h3>
                <ol>
                    <li><strong>Full Sync</strong>: Complete user synchronization</li>
                    <li><strong>Sync Users Only</strong>: Only update user database</li>
                    <li><strong>Test API</strong>: Check API response format</li>
                </ol>
                <p class="error"><strong>Note:</strong> This version fixes the user parsing bug from the previous version.</p>
            </div>
        </div>
    </body>
    </html>';
}
?>